from typing import List
from functools import partial
from .verbs import *


TASK_REGISTRY = {'Simple': 6, 'Normal': 9, 'Difficult': 13}


class Task:

    def __init__(
            self,
            alphabet: List=None,
            ) -> None:
        if alphabet is None:
            self.alphabet = ['a', 'b']
        else:
            self.alphabet = alphabet


class Simple01(Task):
    
    examples = {
            'pos': {
                'query': "Given a word represented by a sequence of letters ('a', 'a', 'b', 'b', 'a', 'a'), does this word belongs to the language?",
                'answer': str({'reason': "The word contains 4 'a's. According to the language's definition, a word belongs to the language if it includes an even number of 'a's. Since 4 is an even number, this word meets the criteria.", 'answer': True})
                },
            'neg': {
                'query': "Given a word represented by a sequence of letters ('a', 'b', 'a', 'b', 'a'), does this word belongs to the language?",
                'answer': str({'reason': "The word contains 3 'a's. The language is defined to include only those words with an even number of 'a's. Since 3 is an odd number, this word does not satisfy the language criteria.", 'answer': False})
                }
        }

    def in_language(self, input_seq: Tuple[str]):
        return has(input_seq, ('a',)) % 2 == 0
    
    def definition(self):
        return 'A language that can be described by a DFA is defined by the following criteria: '\
               'it consists solely of words that contain an even number of '\
               f'a\'s. The alphabet of this language is '\
               f'{self.alphabet}. An empty word epsilon is represented by an empty tuple ().'
    

class Simple02(Task):

    examples = {
            'pos': {
                'query': "Given a word represented by a sequence of letters ('a', 'a', 'a', 'b', 'b'), does this word belongs to the language?",
                'answer': str({'reason': "The word contains 3 'a's. The defined language includes words with an odd number of 'a's. Since 3 is an odd number, this word fits the criteria for the language.", 'answer': True})
                },
            'neg': {
                'query': "Given a word represented by a sequence of letters ('a', 'a', 'b', 'b'), does this word belongs to the language?",
                'answer': str({'reason': "The word contains 2 'a's. According to the language's definition, only words with an odd number of 'a's are included. Since 2 is an even number, this word does not meet the language's requirement.", 'answer': False})
                }
        }

    def in_language(self, input_seq: Tuple[str]):
        return has(input_seq, ('a', )) % 2 == 1
    
    def definition(self):
        return 'A language that can be described by a DFA is defined by the following criteria: '\
               'it consists solely of words that contain an odd number of '\
               f'"{self.alphabet[0]}"s. The alphabet of this language is '\
               f'{self.alphabet}. An empty word epsilon is represented by an empty tuple ().'


class Simple03(Task):
    
    examples = {
            'pos': {
                'query': "Given a word represented by a sequence of letters ('b', 'a', 'a', 'b'), does this word belongs to the language?",
                'answer': str({'reason': "The word ends with the sequence ('a', 'b'). According to the language's definition, words must end with ('a', 'b') to belong. Since this word meets this ending requirement, it is considered part of the language.", 'answer': True})
                },
            'neg': {
                'query': "Given a word represented by a sequence of letters ('a', 'a', 'b', 'b'), does this word belongs to the language?",
                'answer': str({'reason': "The word ends with the sequence ('b', 'b'). The defined language explicitly states that words must end with ('a', 'b') to be included. Because this word ends with ('b', 'b') instead of ('a', 'b'), it does not satisfy the language's condition.", 'answer': False})
                }
        }

    def in_language(self, input_seq: Tuple[str]):
        return end(input_seq, ('a', 'b'))
    
    def definition(self):
        return 'A language that can be described by a DFA is defined by the following criteria: '\
               'it consists solely of words that end with '\
               '(\'a\', \'b\'). The alphabet of this language is '\
               f'{self.alphabet}. An empty word epsilon is represented by an empty tuple ().'
    

class Simple04(Task):
    examples = {
            'pos': {
                'query': "Given a word represented by a sequence of letters ('a', 'b', 'b', 'a', 'a'), does this word belongs to the language?",
                'answer': str({'reason': "The word starts with the letter 'a'. According to the language's definition, a word must begin with 'a' to be included. Since this word adheres to that requirement, it is part of the language.", 'answer': True})
                },
            'neg': {
                'query': "Given a word represented by a sequence of letters ('b', 'a', 'a', 'a', 'a'), does this word belongs to the language?",
                'answer': str({'reason': "The word begins with the letter 'b'. The language's definition clearly states that only words beginning with 'a' are accepted. Therefore, since this word starts with 'b', it does not meet the criteria for inclusion in the language.", 'answer': False})
                }
        }

    def in_language(self, input_seq: Tuple[str]):
        return begin(input_seq, ('a',))
    
    def definition(self):
        return 'A language that can be described by a DFA is defined by the following criteria: '\
               'it consists solely of words that begin with '\
               '"a". The alphabet of this language is '\
               f'{self.alphabet}. An empty word epsilon is represented by an empty tuple ().'
    

class Simple05(Task):

    examples = {
            'pos': {
                'query': "Given a word represented by a sequence of letters ('a', 'a', 'a', 'b', 'a', 'a', 'b'), does this word belongs to the language?",
                'answer': str({'reason': "The word contains the subsequence ('a', 'a', 'b', 'a'), which appears from the third to the sixth letter in the sequence. According to the language's definition, a word must contain this specific subsequence at least once to be considered part of the language. Since this word includes the subsequence, it satisfies the language's requirement.", 'answer': True})
                },
            'neg': {
                'query': "Given a word represented by a sequence of letters ('b', 'a', 'a', 'a', 'a'), does this word belongs to the language?",
                'answer': str({'reason': "The word does not contain the required subsequence ('a', 'a', 'b', 'a'). The language's definition specifies that only words containing this subsequence are included. As this word lacks the sequence, it does not meet the language's criteria and thus does not belong to the language.", 'answer': False})
                }
        }

    def in_language(self, input_seq: Tuple[str]):
        return has(input_seq, ('a', 'a', 'b', 'a')) >= 1
    
    def definition(self):
        return 'A language that can be described by a DFA is defined by the following criteria: '\
               'it consists solely of words that contain '\
               '(\'a\', \'a\', \'b\', \'a\'). The alphabet of this language is '\
               f'{self.alphabet}. An empty word epsilon is represented by an empty tuple ().'
    

class Simple06(Task):

    examples = {
            'pos': {
                'query': "Given a word represented by a sequence of letters ('1', '1', '1', '1', '0'), does this word belongs to the language?",
                'answer': str({'reason':  "The sequence '11110' represents the binary number 11110. When converted to decimal, this is 30, which is an even number. According to the language's definition, a word belongs to the language if it represents an even binary number. Since 30 is even, the word meets the criteria and is included in the language.", 'answer': True})
                },
            'neg': {
                'query': "Given a word represented by a sequence of letters ('0', '1', '0', '0', '0', '1'), does this word belongs to the language?",
                'answer': str({'reason':  "The sequence '010001' corresponds to the binary number 010001, which is 17 in decimal form. Since 17 is an odd number, it does not meet the language's requirement that words must represent even binary numbers. Therefore, this word does not belong to the language.", 'answer': False})
                }
        }
    
    def __init__(self, alphabet: List = None) -> None:
        if alphabet is None:
            super().__init__(['0', '1'])
        else:
            super().__init__(alphabet)

    def in_language(self, input_seq: Tuple[str]):
        return len(input_seq) > 0 and int(''.join(input_seq), 2) % 2 == 0
    
    def definition(self):
        return 'A language that can be described by a DFA is defined by the following specific criteria: '\
            'it consists only of words that represent binary numbers '\
            f'(using the alphabet {self.alphabet}) which are also multiples '\
            'of 2 when converted to their decimal equivalents. An empty word epsilon is represented by an empty tuple ().'
    

class Normal01(Task):
    
    examples = {
            'pos': {
                'query': "Given a word represented by a sequence of letters ('a', 'a', 'b', 'b', 'a', 'a'), does this word belongs to the language?",
                'answer': str({'reason': "This word contains four 'a's and two 'b's. According to the language's definition, a word needs at least three 'a's and at least two 'b's to belong. This word meets both criteria.", 'answer': True})
                },
            'neg': {
                'query': "Given a word represented by a sequence of letters ('a', 'b', 'a', 'a', 'a'), does this word belongs to the language?",
                'answer': str({'reason': "While this word contains four 'a's, which is more than the minimum three required, it only has one 'b'. The language requires at least two 'b's, so this word does not meet all criteria.", 'answer': False})
                }
        }

    def in_language(self, input_seq: Tuple[str]):
        return has(input_seq, ('a',)) >= 3 and has(input_seq, ('b',)) >= 2
    
    def definition(self):
        return 'A language that can be described by a DFA is defined by the following criteria: '\
               'it consists solely of words that have at least three a’s and at least two b’s.'\
               f'The alphabet of this language is {self.alphabet}. An empty word epsilon is represented by an empty tuple ().'
    

class Normal02(Task):
    
    examples = {
            'pos': {
                'query': "Given a word represented by a sequence of letters ('a', 'a', 'b', 'b', 'b'), does this word belongs to the language?",
                'answer': str({'reason': "The word contains exactly two 'a's and three 'b's. It perfectly meets the language's definition of having exactly two 'a's and at least two 'b's.", 'answer': True})
                },
            'neg': {
                'query': "Given a word represented by a sequence of letters ('a', 'a', 'b', 'b', 'a', 'a'), does this word belongs to the language?",
                'answer': str({'reason':  "This word contains four 'a's, exceeding the limit of two 'a's defined by the language. Despite meeting the requirement for 'b's, the excess 'a's disqualify it.", 'answer': False})
                }
        }

    def in_language(self, input_seq: Tuple[str]):
        return has(input_seq, ('a',)) == 2 and has(input_seq, ('b',)) >= 2
    
    def definition(self):
        return 'A language that can be described by a DFA is defined by the following criteria: '\
               'it consists solely of words that have exactly two a’s and at least two b’s.'\
               f'The alphabet of this language is {self.alphabet}. An empty word epsilon is represented by an empty tuple ().'
    

class Normal03(Task):
    
    examples = {
            'pos': {
                'query': "Given a word represented by a sequence of letters ('a', 'a', 'b'), does this word belongs to the language?",
                'answer': str({'reason': "The word has exactly two 'a's (an even number) and one 'b', aligning perfectly with the language's criteria of having an even number of 'a's and one or two 'b's.", 'answer': True})
                },
            'neg': {
                'query': "Given a word represented by a sequence of letters ('a', 'a', 'b', 'b', 'b', 'a', 'a'), does this word belongs to the language?",
                'answer': str({'reason':  "Though this word has an even number of 'a's (four), it contains three 'b's, which exceeds the maximum of two allowed by the language.", 'answer': False})
                }
        }

    def in_language(self, input_seq: Tuple[str]):
        return has(input_seq, ('a',)) % 2 == 0 and has(input_seq, ('b',)) in [1, 2]
    
    def definition(self):
        return 'A language that can be described by a DFA is defined by the following criteria: '\
               'it consists solely of words that have an even number of a’s and one or two b’s.'\
               f'The alphabet of this language is {self.alphabet}. An empty word epsilon is represented by an empty tuple ().'
    

class Normal04(Task):
    
    examples = {
            'pos': {
                'query': "Given a word represented by a sequence of letters ('a', 'b', 'a', 'b', 'b'), does this word belongs to the language?",
                'answer': str({'reason': "This word contains two 'a's, which is an even number, and each 'a' is directly followed by a 'b', meeting both criteria of the language's definition. Thus, the word qualifies for inclusion in the language.", 'answer': True})
                },
            'neg': {
                'query': "Given a word represented by a sequence of letters ('b', 'a', 'b', 'a'), does this word belongs to the language?",
                'answer': str({'reason': "Although this word has two 'a's (an even number), not every 'a' is followed by a 'b'—the second 'a' is not immediately followed by a 'b'. This violates one of the language's criteria, disqualifying the word.", 'answer': False})
                }
        }

    def in_language(self, input_seq: Tuple[str]):
        return has(input_seq, ('a',)) % 2 == 0 and followed(input_seq, ('a',), ('b',))
    
    def definition(self):
        return 'A language that can be described by a DFA is defined by the following criteria: '\
               'it consists solely of words that have an even number of a’s and each a is followed by at least one b.'\
               f'The alphabet of this language is {self.alphabet}. An empty word epsilon is represented by an empty tuple ().'
    

class Normal05(Task):
    
    examples = {
            'pos': {
                'query': "Given a word represented by a sequence of letters ('a', 'a', 'a', 'b'), does this word belongs to the language?",
                'answer': str({'reason': "The word starts with an 'a' and contains only one 'b'. It perfectly aligns with the language's rules that specify starting with an 'a' and having no more than one 'b'. Therefore, the word is included in the language.", 'answer': True})
                },
            'neg': {
                'query': "Given a word represented by a sequence of letters ('a', 'a', 'b', 'b'), does this word belongs to the language?",
                'answer': str({'reason': "While this word starts with an 'a', it contains two 'b's, which exceeds the limit set by the language's definition of having at most one 'b'. Consequently, the word does not meet the language's criteria.", 'answer': False})
                }
        }

    def in_language(self, input_seq: Tuple[str]):
        return begin(input_seq, ('a',)) and has(input_seq, ('b',)) <= 1
    
    def definition(self):
        return 'A language that can be described by a DFA is defined by the following criteria: '\
               'it consists solely of words that start with an a and has at most one b.'\
               f'The alphabet of this language is {self.alphabet}. An empty word epsilon is represented by an empty tuple ().'
    

class Normal06(Task):
    
    examples = {
            'pos': {
                'query': "Given a word represented by a sequence of letters ('a', 'a', 'a', 'b'), does this word belongs to the language?",
                'answer': str({'reason': "This word has three 'a's, which is an odd number, and it ends with a 'b'. Both of these conditions align with the language's criteria for inclusion: having an odd number of 'a's and ending with 'b'. Therefore, this word belongs to the language.", 'answer': True})
                },
            'neg': {
                'query': "Given a word represented by a sequence of letters ('a', 'b', 'b', 'a', 'a'), does this word belongs to the language?",
                'answer': str({'reason':  "Although this word has an odd number of 'a's (three), it does not end with 'b' but ends with 'a' instead. The requirement that a word must end with 'b' is not met, thus excluding it from the language.", 'answer': False})
                }
        }

    def in_language(self, input_seq: Tuple[str]):
        return has(input_seq, ('a',)) % 2 == 1 and end(input_seq, ('b',))
    
    def definition(self):
        return 'A language that can be described by a DFA is defined by the following criteria: '\
               'it consists solely of words that have an odd number of a’s and ends with \"b\".'\
               f'The alphabet of this language is {self.alphabet}. An empty word epsilon is represented by an empty tuple ().'
    

class Normal07(Task):
    
    examples = {
            'pos': {
                'query': "Given a word represented by a sequence of letters ('a', 'a', 'a', 'b'), does this word belongs to the language?",
                'answer': str({'reason': "The word contains four letters, making its length even, and it includes three 'a's, which is an odd number. These characteristics precisely match the language's requirements of having an even length and an odd number of 'a's. Thus, the word is included in the language.", 'answer': True})
                },
            'neg': {
                'query': "Given a word represented by a sequence of letters ('a', 'b', 'b', 'a', 'a'), does this word belongs to the language?",
                'answer': str({'reason': "This word contains five letters, which is an odd length. The language requires words to have even lengths, regardless of having an odd number of 'a's (which this word does meet). The failure to meet the even length requirement disqualifies it from the language.", 'answer': False})
                }
        }

    def in_language(self, input_seq: Tuple[str]):
        return has(input_seq, ('a',)) % 2 == 1 and len(input_seq) % 2 == 0
    
    def definition(self):
        return 'A language that can be described by a DFA is defined by the following criteria: '\
               'it consists solely of words that have even length and an odd number of a’s.'\
               f'The alphabet of this language is {self.alphabet}. An empty word epsilon is represented by an empty tuple ().'
    

class Normal08(Task):
    
    examples = {
            'pos': {
                'query': "Given a word represented by a sequence of letters ('a', 'a', 'a', 'b'), does this word belongs to the language?",
                'answer': str({'reason': "This word begins with 'a' and ends with 'b', fulfilling both criteria set by the language's definition. As it starts with 'a' and concludes with 'b', it is included in the language.", 'answer': True})
                },
            'neg': {
                'query': "Given a word represented by a sequence of letters ('a', 'b', 'b', 'a', 'a'), does this word belongs to the language?",
                'answer': str({'reason': "Although this word starts with 'a', it fails to meet the criteria of ending with 'b' as it ends with 'a' instead. The language's definition requires the word to end with 'b', thus this word does not belong to the language.", 'answer': False})
                }
        }

    def in_language(self, input_seq: Tuple[str]):
        return begin(input_seq, ('a',)) and end(input_seq, ('b',))
    
    def definition(self):
        return 'A language that can be described by a DFA is defined by the following criteria: '\
               'it consists solely of words that start with a and end with b.'\
               f'The alphabet of this language is {self.alphabet}. An empty word epsilon is represented by an empty tuple ().'
    

class Normal09(Task):

    examples = {
            'pos': {
                'query': "Given a word represented by a sequence of letters ('1', '1', '1', '0', '0'), does this word belongs to the language?",
                'answer': str({'reason': "The binary sequence '11100' converts to 28 in decimal, which is a multiple of 4. Therefore, it meets the criteria of the language, which requires binary numbers to be multiples of 4. As such, this word is part of the language.", 'answer': True})
                },
            'neg': {
                'query': "Given a word represented by a sequence of letters ('0', '1', '0', '0', '0', '1'), does this word belongs to the language?",
                'answer': str({'reason': "The binary sequence '010001' converts to 17 in decimal, which is not a multiple of 4. According to the language's definition, only binary numbers that are multiples of 4 qualify. Thus, this word does not meet the required criterion.", 'answer': False})
                }
        }
    
    def __init__(self, alphabet: List = None) -> None:
        if alphabet is None:
            super().__init__(['0', '1'])
        else:
            super().__init__(alphabet)

    def in_language(self, input_seq: Tuple[str]):
        return len(input_seq) > 0 and int(''.join(input_seq), 2) % 4 == 0
    
    def definition(self):
        return 'A language that can be described by a DFA is defined by the following specific criteria: '\
            'it consists only of words that represent binary numbers '\
            f'(using the alphabet {self.alphabet}) which are also multiples '\
            'of 4 when converted to their decimal equivalents. An empty word epsilon is represented by an empty tuple ().'


class Difficult01(Task):

    examples = {
            'pos': {
                'query': "Given a word represented by a sequence of letters ('b', 'a', 'a', 'a'), does this word belongs to the language?",
                'answer': str({'reason': "This word consists of the letters 'b', followed by three 'a's. There is no occurrence of the substring 'ab', which is a sequence of 'a' followed directly by 'b'. Since the language excludes any word containing 'ab', this word adheres to the criteria and is included in the language.", 'answer': True})
                },
            'neg': {
                'query': "Given a word represented by a sequence of letters ('a', 'b', 'b', 'a', 'a'), does this word belongs to the language?",
                'answer': str({'reason': "This word includes the substring 'ab' early in its sequence. The presence of 'ab' directly violates the language's rule that excludes any word containing this specific substring. Therefore, this word does not belong to the language.", 'answer': False})
                }
        }

    def in_language(self, input_seq: Tuple[str]):
        return has(input_seq, ('a', 'b')) == 0
    
    def definition(self):
        return 'A language that can be described by a DFA is defined by the following criteria: '\
               'it consists solely of words that do not contain the substring (\'a\', \'b\').'\
               f'The alphabet of this language is {self.alphabet}. An empty word epsilon is represented by an empty tuple ().'
    

class Difficult02(Task):

    examples = {
            'pos': {
                'query': "Given a word represented by a sequence of letters ('b', 'a', 'a', 'a'), does this word belongs to the language?",
                'answer': str({'reason': "The sequence of this word is 'b', followed by three 'a's, with no occurrence of the substring 'baba'. Since the definition explicitly excludes words containing 'baba', and this word does not feature that substring, it fully complies with the language's requirements.", 'answer': True})
                },
            'neg': {
                'query': "Given a word represented by a sequence of letters ('a', 'b', 'b', 'a', 'b', 'a', 'b'), does this word belongs to the language?",
                'answer': str({'reason': "This word contains the substring 'baba', starting from the second 'b' in the sequence. This directly contravenes the language's prohibition against the substring 'baba'. Consequently, this word is excluded from the language due to containing the disallowed substring.", 'answer': False})
                }
        }

    def in_language(self, input_seq: Tuple[str]):
        return has(input_seq, ('b', 'a', 'b', 'a')) == 0
    
    def definition(self):
        return 'A language that can be described by a DFA is defined by the following criteria: '\
               'it consists solely of words that do not contain the substring (\'b\', \'a\', \'b\', \'a\').'\
               f'The alphabet of this language is {self.alphabet}. An empty word epsilon is represented by an empty tuple ().'
    

class Difficult03(Task):

    examples = {
            'pos': {
                'query': "Given a word represented by a sequence of letters ('a', 'a', 'a', 'a'), does this word belongs to the language?",
                'answer': str({'reason': "This word is composed solely of the letter 'a' and does not include the substrings 'ab' or 'ba'. The language's definition excludes any word containing these substrings. Since none are present, this word adheres to the criteria and belongs to the language.", 'answer': True})
                },
            'neg': {
                'query': "Given a word represented by a sequence of letters ('a', 'a', 'a', 'b'), does this word belongs to the language?",
                'answer': str({'reason': "This word ends with the substring 'ab'. Since the language explicitly prohibits words containing 'ab' or 'ba', the presence of 'ab' disqualifies this word from the language.", 'answer': False})
                }
        }

    def in_language(self, input_seq: Tuple[str]):
        return has(input_seq, ('a', 'b')) == 0 and has(input_seq, ('b', 'a')) == 0
    
    def definition(self):
        return 'A language that can be described by a DFA is defined by the following criteria: '\
               'it consists solely of words that contain neither the substrings (\'a\', \'b\') nor (\'b\', \'a\').'\
               f'The alphabet of this language is {self.alphabet}. An empty word epsilon is represented by an empty tuple ().'
    

class Difficult04(Task):

    examples = {
            'pos': {
                'query': "Given a word represented by a sequence of letters ('a', 'b', 'a', 'b'), does this word belongs to the language?",
                'answer': str({'reason': "This word, consisting of the pattern 'abab', does not fit the excluded pattern `a*b*`, where all 'a's would precede all 'b's without interruption. Since the language only excludes words that strictly follow `a*b*`, this word's alternating pattern of 'a's and 'b's qualifies it for inclusion in the language.", 'answer': True})
                },
            'neg': {
                'query': "Given a word represented by a sequence of letters ('a', 'a', 'a', 'b', 'b', 'b'), does this word belongs to the language?",
                'answer': str({'reason': "This word strictly follows the pattern `a*b*`, where all 'a's come before any 'b's. The language definition excludes words matching this specific pattern. Therefore, since this word adheres to the excluded pattern, it does not belong to the language.", 'answer': False})
                }
        }

    def in_language(self, input_seq: Tuple[str]):
        pattern = r'^a*b*$'
        match = re.fullmatch(pattern, ''.join(input_seq))
        return match is None
    
    def definition(self):
        return 'A language that can be described by a DFA is defined by the following criteria: '\
               'it consists solely of words that are any string not in a∗b∗.'\
               f'The alphabet of this language is {self.alphabet}. An empty word epsilon is represented by an empty tuple ().'
    

class Difficult05(Task):

    examples = {
            'pos': {
                'query': "Given a word represented by a sequence of letters ('a', 'a', 'a', 'b', 'b', 'b'), does this word belongs to the language?",
                'answer': str({'reason': "This word contains three consecutive 'a's followed by three consecutive 'b's, which does not match the `(ab+)*` pattern where each 'a' must be directly followed by at least one 'b'. As the word does not follow this sequence, it does not conform to the excluded pattern, thus it is included in the language.", 'answer': True})
                },
            'neg': {
                'query': "Given a word represented by a sequence of letters ('a', 'b', 'a', 'b'), does this word belongs to the language?",
                'answer': str({'reason': "The sequence of this word matches the `(ab+)*` pattern exactly, as each 'a' is followed directly by a 'b'. Since the language excludes words that strictly adhere to this pattern, this word does not qualify for inclusion in the language.", 'answer': False})
                }
        }

    def in_language(self, input_seq: Tuple[str]):
        pattern = r'^(ab+)*$'
        match = re.fullmatch(pattern, ''.join(input_seq))
        return match is None
    
    def definition(self):
        return 'A language that can be described by a DFA is defined by the following criteria: '\
               'it consists solely of words that are any string not in (ab+)∗.'\
               f'The alphabet of this language is {self.alphabet}. An empty word epsilon is represented by an empty tuple ().'
    

class Difficult06(Task):

    examples = {
            'pos': {
                'query': "Given a word represented by a sequence of letters ('a', 'a', 'a', 'b', 'b'), does this word belongs to the language?",
                'answer': str({'reason': "This word contains a mix of 'a's followed by 'b's, breaking the pattern of `a* ∪ b*`, which includes strings made entirely of 'a's or entirely of 'b's. Because it contains both letters and does not fit into the excluded patterns, this word belongs to the language.", 'answer': True})
                },
            'neg': {
                'query': "Given a word represented by a sequence of letters ('a', 'a', 'a'), does this word belongs to the language?",
                'answer': str({'reason': "The sequence is composed entirely of 'a's, which fits the `a*` pattern. The language excludes any word that can be described by `a* ∪ b*`. Therefore, this word, adhering to one of these specific patterns, does not qualify for inclusion in the language.", 'answer': False})
                }
        }

    def in_language(self, input_seq: Tuple[str]):
        pattern = r'^a*$|^b*$'
        match = re.fullmatch(pattern, ''.join(input_seq))
        return match is None
    
    def definition(self):
        return 'A language that can be described by a DFA is defined by the following criteria: '\
               'it consists solely of words that are any string not in a* ∪ b*.'\
               f'The alphabet of this language is {self.alphabet}. An empty word epsilon is represented by an empty tuple ().'
    

class Difficult07(Task):

    examples = {
            'pos': {
                'query': "Given a word represented by a sequence of letters ('a', 'a', 'a', 'b', 'b'), does this word belongs to the language?",
                'answer': str({'reason': "This word contains three 'a's. The language's definition specifies exclusion of words that contain exactly two 'a's. Since this word does not meet the condition of having exactly two 'a's, it belongs to the language.", 'answer': True})
                },
            'neg': {
                'query': "Given a word represented by a sequence of letters ('a', 'b', 'b', 'a'), does this word belongs to the language?",
                'answer': str({'reason': "This word contains exactly two 'a's. According to the language's definition, any word with exactly two 'a's is excluded. Therefore, this word does not qualify for inclusion in the language.", 'answer': False})
                }
        }

    def in_language(self, input_seq: Tuple[str]):
        return has(input_seq, ('a',)) != 2
    
    def definition(self):
        return 'A language that can be described by a DFA is defined by the following criteria: '\
               'it consists solely of words that does not contain exactly two a’s.'\
               f'The alphabet of this language is {self.alphabet}. An empty word epsilon is represented by an empty tuple ().'
    

class Difficult08(Task):

    examples = {
            'pos': {
                'query': "Given a word represented by a sequence of letters ('a', 'a', 'a', 'b', 'b'), does this word belongs to the language?",
                'answer': str({'reason': "The word consists of a combination of 'a's and 'b's and does not represent a single 'a' or 'b'. The definition excludes only the singular instances of 'a' or 'b'. Since this word contains a mix, it does not match the exclusion criteria and thus belongs to the language.", 'answer': True})
                },
            'neg': {
                'query': "Given a word represented by a sequence of letters ('a',), does this word belongs to the language?",
                'answer': str({'reason': "This word is composed of a single 'a'. The language's definition explicitly excludes words that are solely 'a' or 'b'. Therefore, this word does not meet the criteria for inclusion.", 'answer': False})
                }
        }

    def in_language(self, input_seq: Tuple[str]):
        return input_seq != ('a',) and input_seq != ('b',)
    
    def definition(self):
        return 'A language that can be described by a DFA is defined by the following criteria: '\
               'it consists solely of words that are any sequence except (\'a\',) and (\'b\',).'\
               f'The alphabet of this language is {self.alphabet}. An empty word epsilon is represented by an empty tuple ().'
    

class Difficult09(Task):

    examples = {
            'pos': {
                'query': "Given a word represented by a sequence of letters ('a', 'a', 'a', 'b', 'b'), does this word belongs to the language?",
                'answer': str({'reason': "This word is five characters long and the third character is 'a'. According to the language's definition, any word that is at least three characters long and has 'a' as the third character qualifies. This word meets both conditions and thus belongs to the language.", 'answer': True})
                },
            'neg': {
                'query': "Given a word represented by a sequence of letters ('a', 'a', 'b'), does this word belongs to the language?",
                'answer': str({'reason': "While this word is three characters long, the third character is 'b' instead of 'a'. The language's definition specifically requires the third symbol to be 'a' for a word to qualify. Since this condition is not met, the word does not belong to the language.", 'answer': False})
                }
        }

    def in_language(self, input_seq: Tuple[str]):
        return len(input_seq) >= 3 and input_seq[2] == 'a'
    
    def definition(self):
        return 'A language that can be described by a DFA is defined by the following criteria: '\
               'it consists solely of words that have length at least 3 and its third symbol is an a.'\
               f'The alphabet of this language is {self.alphabet}. An empty word epsilon is represented by an empty tuple ().'
    

class Difficult10(Task):

    examples = {
            'pos': {
                'query': "Given a word represented by a sequence of letters ('a', 'a', 'a', 'b', 'b'), does this word belongs to the language?",
                'answer': str({'reason': "The word starts with 'a' and is five characters long, which is an odd number. The language includes words that start with 'a' and have an odd length. This word fits this description perfectly, hence it belongs to the language.", 'answer': True})
                },
            'neg': {
                'query': "Given a word represented by a sequence of letters ('a', 'a', 'b', 'b'), does this word belongs to the language?",
                'answer': str({'reason': "This word begins with 'a' but is four characters long, which is an even number. The language specifies that words starting with 'a' must have an odd length to qualify. As this word does not meet this requirement, it does not belong to the language.", 'answer': False})
                }
        }

    def in_language(self, input_seq: Tuple[str]):
        return (begin(input_seq, ('a',)) and len(input_seq) % 2 == 1) or (begin(input_seq, ('b',)) and len(input_seq) % 2 == 0)
    
    def definition(self):
        return 'A language that can be described by a DFA is defined by the following criteria: '\
               'it consists solely of words that start with a and have odd length, or start with b and have even length.'\
               f'The alphabet of this language is {self.alphabet}. An empty word epsilon is represented by an empty tuple ().'
    

class Difficult11(Task):

    examples = {
            'pos': {
                'query': "Given a word represented by a sequence of letters ('b', 'a', 'b', 'b', 'b'), does this word belongs to the language?",
                'answer': str({'reason': "In this word, characters in odd positions (1st, 3rd, 5th) are all 'b's. The language's definition requires that every odd position of a word must be 'b'. Since this word meets that condition, it is included in the language.", 'answer': True})
                },
            'neg': {
                'query': "Given a word represented by a sequence of letters ('a', 'a', 'b', 'b'), does this word belongs to the language?",
                'answer': str({'reason':  "The first character, which is an odd position, is 'a' instead of 'b'. The language excludes any word that does not have 'b' at every odd position. Thus, this word does not comply with the language's criteria and is not part of the language.", 'answer': False})
                }
        }

    def in_language(self, input_seq: Tuple[str]):
        for i in range(0, len(input_seq), 2):
            if input_seq[i] != 'b':
                return False
        return True
    
    def definition(self):
        return 'A language that can be described by a DFA is defined by the following criteria: '\
               'it consists solely of words that every odd position of them is \"b\".'\
               f'The alphabet of this language is {self.alphabet}. An empty word epsilon is represented by an empty tuple ().'
    

class Difficult12(Task):

    examples = {
            'pos': {
                'query': "Given a word represented by a sequence of letters ('1', '1', '1', '1'), does this word belongs to the language?",
                'answer': str({'reason': "The binary sequence '1111' converts to 15 in decimal, which is a multiple of 3. According to the language's definition, a word must represent a binary number that is a multiple of 3 to qualify. Since 15 is a multiple of 3, this word meets the criteria and is included in the language.", 'answer': True})
                },
            'neg': {
                'query': "Given a word represented by a sequence of letters ('1', '1', '0', '0', '1'), does this word belongs to the language?",
                'answer': str({'reason': "The binary sequence '11001' converts to 25 in decimal, which is not a multiple of 3. The language requires that words represent binary numbers that are multiples of 3. Since 25 is not a multiple of 3, this word does not qualify for inclusion in the language.", 'answer': False})
                }
        }
    
    def __init__(self, alphabet: List = None) -> None:
        if alphabet is None:
            super().__init__(['0', '1'])
        else:
            super().__init__(alphabet)

    def in_language(self, input_seq: Tuple[str]):
        return len(input_seq) > 0 and int(''.join(input_seq), 2) % 3 == 0
    
    def definition(self):
        return 'A language that can be described by a DFA is defined by the following specific criteria: '\
            'it consists only of words that represent binary numbers '\
            f'(using the alphabet {self.alphabet}) which are also multiples '\
            'of 3 when converted to their decimal equivalents. An empty word epsilon is represented by an empty tuple ().'
    

class Difficult13(Task):

    examples = {
            'pos': {
                'query': "Given a word represented by a sequence of letters ('1', '1', '1', '1'), does this word belongs to the language?",
                'answer': str({'reason': "The binary sequence '1111' converts to 15 in decimal. Since 15 is a multiple of 5, this word meets the criteria set forth by the language's definition that includes only binary numbers that are multiples of 5. Therefore, the word qualifies for inclusion in the language.", 'answer': True})
                },
            'neg': {
                'query': "Given a word represented by a sequence of letters ('1', '1', '0', '1', '1'), does this word belongs to the language?",
                'answer': str({'reason': "The binary sequence '11011' converts to 27 in decimal. 27 is not a multiple of 5. According to the language's definition, only binary numbers that are multiples of 5 qualify. Since 27 is not a multiple of 5, this word does not meet the criteria and is correctly excluded from the language.", 'answer': False})
                }
        }
    
    def __init__(self, alphabet: List = None) -> None:
        if alphabet is None:
            super().__init__(['0', '1'])
        else:
            super().__init__(alphabet)

    def in_language(self, input_seq: Tuple[str]):
        return len(input_seq) > 0 and int(''.join(input_seq), 2) % 5 == 0
    
    def definition(self):
        return 'A language that can be described by a DFA is defined by the following specific criteria: '\
            'it consists only of words that represent binary numbers '\
            f'(using the alphabet {self.alphabet}) which are also multiples '\
            'of 5 when converted to their decimal equivalents. An empty word epsilon is represented by an empty tuple ().'
